// +-----------------------------------------------------------------------+
// |                                                                       |
// | NN search in a LB-tree constructed by using agglomerative clustering. |
// |                                                                       |
// + - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - +
// |                                                                       |
// | Author: Yong-Sheng Chen (yschen@iis.sinica.edu.tw)      10/10/00      |
// |         Institute of Information Science                              |
// |         Academia Sinica, Taipei, Taiwan                               |
// |                                                                       |
// +-----------------------------------------------------------------------+

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "cluster.h"


MEAN_NODE *root=NULL;
MEAN_NODE **heap=NULL;

double *qpoint;
int dim, pnum;

extern void transform(double *, double *);


// read the tree structure from a file
int read_structure(char *fname)
{
   int i, tmpv, nsize, dsize, ncount, tdim;
   FILE *fp;
   MEAN_NODE *mnp;
   double *points;
 
   fp = fopen(fname, "rb");
   if (fp==NULL)
   {
      fprintf(stderr, "read_structure(): file %s open error!\n", fname);
      return 0;
   }
   fread(&dim, sizeof(int), 1, fp);   // dimension of each data point
   fread(&pnum, sizeof(int), 1, fp);  // #sample points
   fread(&nsize, sizeof(int), 1, fp); // #mean nodes
   fread(&dsize, sizeof(int), 1, fp); // total dimensions for all mean nodes

   initialize_transformation(dim);

   heap = (MEAN_NODE **)malloc(sizeof(MEAN_NODE *)*pnum);
   qpoint = (double *)malloc(sizeof(double)*dim);
   root=(MEAN_NODE *)malloc(sizeof(MEAN_NODE)*nsize);
   points=(double *)malloc(sizeof(double)*dsize);

   if ( heap==NULL || qpoint==NULL || root==NULL || points==NULL )
   {
      fprintf(stderr, "read_structure(): malloc error.\n");
      return 0;
   }

   fread(root, sizeof(MEAN_NODE), nsize, fp);
   fread(points, sizeof(double), dsize, fp);

   fclose(fp);

   // Reallocate the address
   tdim=0; ncount=1;
   for (i=0; i<nsize; i++)
   {
      mnp=&root[i];
      if (mnp->ndim!=tdim)
      {
	 printf("#nodes at the level (dim %3d): %6d\n", tdim, ncount);
         tdim=mnp->ndim;
	 ncount=1;
      }
      else
         ncount++;
      mnp->data = points+(int)mnp->data;
      if (mnp->sibling)
         mnp->sibling = root+(int)mnp->sibling;
      if (mnp->ndim!=dim)   // leaf's child is the index, no need to reallocate
         mnp->child = root+(int)mnp->child;
   }
   printf("#nodes at the level (dim %3d): %6d\n", tdim, ncount);

   return(pnum);
}


// perform the inquiry for the input data point
void inquire(double *point, int *index, double *dist)
{
   int hsize, nindex, tdim, i, j, k;
   double ndist, tdist, tmpv;
   MEAN_NODE *mnp, *v;

   transform(point, qpoint);   // transform the data point

   // heap construction for the first level
   for (mnp=root, hsize=0; mnp!=NULL; mnp=mnp->sibling, hsize++)
   {
      mnp->dist=fabs(qpoint[0]-mnp->data[0])-mnp->radius;
      heap[hsize]=mnp;
   }
   // downheap to let the top node having smallest distance
   for (k=hsize/2-1; k>=0; k--)
   {
      i = k;
      v = heap[k];
      tmpv=v->dist;
      while (i<hsize/2)
      {
         j = i+i+1;
         if (j < hsize-1 && heap[j]->dist > heap[j+1]->dist) j++;
         if (tmpv<=heap[j]->dist) break;
         heap[i] = heap[j];
         i = j;
      }
      heap[i]=v;
   }


   // perform the winner-update search strategy (best-first search)
   while (heap[0]->ndim < dim)
   {
      //update the top node
      mnp = heap[0]->child;
      tdim = mnp->ndim;
      if (heap[0]->radius==0.0)    // can accumulate the distance
      {
	 tdist=heap[0]->dist*heap[0]->dist;
         for (i=heap[0]->ndim; i<tdim; i++)
         {
            tmpv = qpoint[i]-mnp->data[i];
	    tdist += tmpv*tmpv;
         }
      }
      else                         // have to calculate distance all overagain
      {
         tdist=0;
         for (i=0; i<tdim; i++)
         {
            tmpv = qpoint[i]-mnp->data[i];
	    tdist += tmpv*tmpv;
         }
      }
      mnp->dist = sqrt(tdist)-mnp->radius;

      // downheap the top node
      i = 0;
      while (i<hsize/2)
      {
         j = i+i+1;
         if (j < hsize-1 && heap[j]->dist > heap[j+1]->dist) j++;
         if (mnp->dist<=heap[j]->dist) break;
         heap[i] = heap[j];
         i = j;
      }
      heap[i]=mnp;

      // add the siblings
      for (mnp=mnp->sibling; mnp!=NULL; mnp=mnp->sibling)
      {
         tdist=0;
         for (i=0; i<tdim; i++)
         {
	    tmpv = qpoint[i]-mnp->data[i];
	    tdist += tmpv*tmpv;
         }
         mnp->dist = sqrt(tdist)-mnp->radius;

	 // upheap
         i=hsize++;
	 while (i>0)
	 {
	    j=(i-1)/2;
	    if (mnp->dist>=heap[j]->dist) break;
            heap[i] = heap[j];
            i = j;
	 }
         heap[i]=mnp;
      }
   }

// return the result at the top node
   *index = (int) heap[0]->child;
   *dist = heap[0]->dist;
}
