// +-----------------------------------------------------------------------+
// |                                                                       |
// | Given dimension and # points, this program generates each coordinate  |
// | for each point by using uniformly or normally distributed random      |
// | process.                                                              |
// |                                                                       |
// + - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - +
// |                                                                       |
// | Author: Yong-Sheng Chen (yschen@iis.sinica.edu.tw)      10/14/99      |
// |         Institute of Information Science                              |
// |         Academia Sinica, Taipei, Taiwan                               |
// |                                                                       |
// +-----------------------------------------------------------------------+

#include <stdio.h>
#include <stdlib.h>
#include <values.h>
#include <math.h>
#include <sys/timeb.h>


// Set the seed for the uniform distribution random number generator
void my_srand()
{
   struct timeb timeseed;   // seed for random number generator

   ftime(&timeseed);
   srandom(timeseed.millitm);
}


// Generate uniformly distributed random number over [-1, 1]
double my_drand()
{
   double d;

   d=(double)random()/RAND_MAX;       // d: [0.0, 1.0]
   return 2.0*(d-0.5);
}


// Generate normally distributed random number with mean 0 and
// standard deviation 1
double my_nrand(void){
   static int iset=0;
   static double gset;
   double fac,r,v1,v2;

   if (iset == 0) {
      do {
         v1=my_drand();
         v2=my_drand();
         r=v1*v1+v2*v2;
      } while (r >= 1.0 || r == 0.0);
      fac=sqrt(-2.0*log(r)/r);
      gset=v1*fac;
      iset=1;
      return v2*fac;
   }
   else {
      iset=0;
      return gset;
   }
}


int main(int argc, char *argv[])
{
   FILE *fp;
   int type, dim, pnum, d, p;
   double *point;

   if (argc != 5)
   {
      fprintf(stderr, "Usage: %s type fname dim #points\n", argv[0]);
      fprintf(stderr, "\t  type 0: random signal\n");
      fprintf(stderr, "\t  type 1: autocorrelated signal\n");
      fprintf(stderr, "\t   fname: output file of the generated point set\n");
      fprintf(stderr, "\t     dim: dimension of the underlying space\n");
      fprintf(stderr, "\t #points: number of points to be generated\n");
      exit(1);
   }

   type = atoi(argv[1]);
   fp   = fopen(argv[2],"wb");
   dim  = atoi(argv[3]);
   pnum = atoi(argv[4]);

   fprintf(fp, "POINTS %d %d\n", dim, pnum);
   fprintf(stdout, "Generating %d random points with %d dimension...\n",
	   pnum, dim);

   point = (double *)malloc(sizeof(double)*dim);

   my_srand();
   for (p=0; p<pnum; p++)
   {
      switch (type)
      {
         case 0:
            for (d=0; d<dim; d++)
               point[d] = my_drand();
	    break;
         case 1:
            point[0] = my_drand();
            for (d=1; d<dim; d++)
               {
	          point[d] = point[d-1]+my_nrand()*0.1;
                  if (point[d]<-1.0)
                     point[d]=-1.0;
                  if (point[d]>1.0)
                     point[d]=1.0;
               }
	    break;
         default:
	    fprintf(stderr, "type unknown!\n");
      }
      fwrite(point, sizeof(double), dim, fp);
   }

   fprintf(stdout, "Done.\n\n");

   fclose(fp);
}
