// +-----------------------------------------------------------------------+
// |                                                                       |
// | Nearest neighbor search by using Winner-Update strategy               |
// |                                                                       |
// | Ref.: Y.-S. Chen, Y.-P. Hung, and C.-S. Fuh, "Winner-Update Algorithm |
// |       for Nearest Neighbor Search", ICPR2000, vol.2, pp. 708--711.    |
// |                                                                       |
// + - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - +
// |                                                                       |
// | Author: Yong-Sheng Chen (yschen@iis.sinica.edu.tw)      10/14/99      |
// |         Institute of Information Science                              |
// |         Academia Sinica, Taipei, Taiwan                               |
// |                                                                       |
// +-----------------------------------------------------------------------+

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <values.h>

typedef struct {
   int      i;       // index in pointsdb
   int      level;   // level of the layer under processing
   double   dist;    // temporary distance
   } POINT_NODE;

POINT_NODE *heap=NULL;
int *hindex=NULL;
int *pindex=NULL;
double *pointsdb, *qpoint;
int dim, pnum, maxLevel, psize;
int *mdim;


void ConstructMultilayer(double *ps, double *pd)
{
   int i, j;
   double *p1, *p2, t1, t2;

   memcpy(pd+dim-1, ps, sizeof(double)*dim);
   for (i=maxLevel; i>0; i--)
   {
      p1 = pd+mdim[i]-1;
      p2 = pd+mdim[i-1]-1;
      for (j=0; j<mdim[i-1]; j++)
      {
         t1 = *p1++;
         t2 = *p1++;
         *p2++ = sqrt(t1*t1+t2*t2);
      }
   }
}


int read_structure(char *fname)
{
   int i, tmpv, levelNumber;
   FILE *fp;

   fp = fopen(fname, "rb");
   if (fp==NULL)
   {
      fprintf(stderr, "read_structure(): file %s open error!\n", fname);
      return 0;
   }
   fread(&dim, sizeof(int), 1, fp);
   fread(&pnum, sizeof(int), 1, fp);
   psize = 2*dim-1;

   for (maxLevel=0, tmpv=1; tmpv<dim; maxLevel++)  // check dimension
      tmpv *= 2;
   if (tmpv != dim)
   {
      fprintf(stderr, "This program can only deal with 2^n dimension.\n");
      return 0;
   }
   levelNumber = maxLevel+1;

   mdim = (int *)malloc(sizeof(int)*levelNumber);
   heap = (POINT_NODE *)malloc(sizeof(POINT_NODE)*pnum);
   hindex = (int *)malloc(sizeof(int)*pnum);
   pindex = (int *)malloc(sizeof(int)*pnum);
   pointsdb = (double *)malloc(sizeof(double)*pnum*psize);
   qpoint = (double *)malloc(sizeof(double)*psize);
   if (mdim==NULL || heap==NULL || hindex==NULL || pindex==NULL || 
       pointsdb==NULL || qpoint == NULL)
   {
      fprintf(stderr, "read_structure(): malloc error.\n");
      return 0;
   }

   fread(pindex, sizeof(int), pnum, fp);
   fread(pointsdb, sizeof(double), pnum*psize, fp);

   fclose(fp);

   for (i=1, mdim[0]=1; i<levelNumber; i++)
      mdim[i] = mdim[i-1]*2;

   return(pnum);
}


static int left, right;

void binsearch(int *index, double *dist)
{
   int l=0, r=pnum-1, x;
   double key=qpoint[0], distl, distr;

   while (r>=l)
   {
      x = (l+r)/2;
      if (key < *(pointsdb+pindex[x]*psize))
	 r=x-1;
      else
	 l=x+1;
   }
   if (r<0) r=0;
   if (l>=pnum) l=pnum-1;
   distl = fabs(key-(*(pointsdb+pindex[l]*psize)));
   distr = fabs(key-(*(pointsdb+pindex[r]*psize)));
   if (distl < distr)
   {
      *index = pindex[l];
      *dist  = distl;
      left = l-1;
      right = l+1;
   }
   else
   {
      *index = pindex[r];
      *dist  = distr;
      left = r-1;
      right = r+1;
   }
}

void nextsearch(int *index, double *dist)
{
   double key=qpoint[0], distl, distr;

   if (left>=0)
   {
      distl = fabs(key-*(pointsdb+pindex[left]*psize));
      if (right<pnum)
      {
         distr = fabs(key-*(pointsdb+pindex[right]*psize));
         if (distl < distr)
         {
            *index = pindex[left];
            *dist  = distl;
            left--;
         }
         else
         {
            *index = pindex[right];
            *dist  = distr;
            right++;
         }
      }
      else
      {
         *index = pindex[left];
         *dist  = distl;
         left--;
      }
   }
   else
   {
      if (right<pnum)
      {
         distr = fabs(key-*(pointsdb+pindex[right]*psize));
         *index = pindex[right];
         *dist  = distr;
         right++;
      }
      else
      {
	 *dist = MAXDOUBLE;
      }
   }
}


void inquire(double *point, int *index, double *dist)
{
   int hsize, nindex, level, tdim, i, j, k, v;
   double ndist, tdist, tmpv;
   POINT_NODE *node;
   double *p1, *p2;

   ConstructMultilayer(point, qpoint);

   node = heap;
   hindex[0]=0;
   hsize=1;
   node->level=0;
   binsearch(&(node->i), &(node->dist));
   nextsearch(&nindex, &ndist);


   while (node->level < maxLevel)
   {
      level = ++(node->level);
      tdim = mdim[level];
      p1 = qpoint+tdim-1;
      p2 = pointsdb+(node->i)*psize+tdim-1;
      tdist=0;
      for (i=0; i<tdim; i++, p1++, p2++)
      {
	 tmpv = *p1-*p2;
	 tmpv *= tmpv;
	 tdist += tmpv;
      }
      node->dist = sqrt(tdist);

   // downheap
      i = 0;
      v = hindex[0];
      while (i<hsize/2)
      {
         j = i+i+1;
         if (j < hsize-1 && heap[hindex[j]].dist > heap[hindex[j+1]].dist) j++;
         if (node->dist<=heap[hindex[j]].dist) break;
         hindex[i] = hindex[j];
         i = j;
      }
      hindex[i]=v;
      node = &heap[hindex[0]];

      if (node->dist > ndist)
      {
	 node = &heap[hsize];
	 node->i = nindex;
	 node->level = 0;
	 node->dist = ndist;
         nextsearch(&nindex, &ndist);
         // insert
	 for (k=hsize; k>0; k=(k-1)/2)
	    hindex[k] = hindex[(k-1)/2];
         hindex[0] = hsize++;
      }
   }

// return the result
   node = &heap[hindex[0]];
   *index = node->i;
   *dist = node->dist;
}
