/*
 * @(#)hprof_object.c	1.5 98/12/17
 *
 * Copyright 1997, 1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

#include "hprof.h"

/* The table that maps objects to allocation sites */
#define HPROF_OBJMAP_TABLE_SIZE 200003
static hprof_objmap_t **hprof_objmap_table;
static hprof_objmap_t *objmap_free_list = NULL;

void hprof_objmap_init(void)
{
    hprof_objmap_table = 
        hprof_calloc(HPROF_OBJMAP_TABLE_SIZE * sizeof(hprof_objmap_t *));
}   

static void add_alloc_stats(hprof_site_t *site, int size)
{
     site->n_alloced_instances += 1;
     site->n_live_instances += 1;
     site->n_alloced_bytes += size;
     site->n_live_bytes += size;
     site->changed = 1;
     
     total_alloced_bytes = jlong_add(total_alloced_bytes, jint_to_jlong(size));
     total_alloced_instances = jlong_add(total_alloced_instances, jint_to_jlong(1));
     total_live_bytes += size;
     total_live_instances += 1;
}

static void sub_alloc_stats(hprof_site_t *site, int size)
{
    site->n_live_instances--;
    site->n_live_bytes -= size;
    site->changed = 1;
    total_live_instances--;
    total_live_bytes -= size;
}
   
	
/* The objmap table maps objects to where they are allocated. Since this
 * is a much bigger table than site or trace tables, we do not use the
 * generic hash table routines for this purpose. Instead, we write
 * specialized code for better performance.
 */
static hprof_objmap_t *hprof_objmap_add(jobjectID obj_id, jint arena_id,
					hprof_site_t *site, int size)
{
    int index = (unsigned long)obj_id % HPROF_OBJMAP_TABLE_SIZE;
    hprof_objmap_t *bucket;
    if (objmap_free_list) {
        bucket = objmap_free_list;
	objmap_free_list = bucket->next;
    } else {
        bucket = hprof_calloc(sizeof(hprof_objmap_t));
    }
    bucket->size = size;
    bucket->site = site;
    bucket->obj_id = obj_id;
    bucket->arena_id = arena_id;
    bucket->next = hprof_objmap_table[index];
    hprof_objmap_table[index] = bucket;
        
    if (site != NULL) {
        add_alloc_stats(site , size);
    }
    return bucket;
}

static void hprof_objmap_del(jobjectID obj_id)
{
    int index = (unsigned long)obj_id % HPROF_OBJMAP_TABLE_SIZE;
    hprof_objmap_t **p = hprof_objmap_table + index;
    hprof_objmap_t *bucket;
    while ((bucket = *p)) {
        if (bucket->obj_id == obj_id) {
	    *p = bucket->next;
	    bucket->next = objmap_free_list;
	    objmap_free_list = bucket;
	    if (bucket->site != NULL) {
	        sub_alloc_stats(bucket->site, bucket->size);
	    }
	    return;
	}
	p = &(bucket->next);
    }
    return;
}

static void hprof_objmap_del_arena(jint arena_id)
{
    int i;
    for (i = 0; i < HPROF_OBJMAP_TABLE_SIZE; i++) {
        hprof_objmap_t **p = hprof_objmap_table + i;
	hprof_objmap_t *bucket;
	while ((bucket = *p)) {
	    if (bucket->arena_id == arena_id) {
	        *p = bucket->next;
		bucket->next = objmap_free_list;
		objmap_free_list = bucket;
		if ((bucket->site != NULL) && (bucket->size != 0)) {
		    sub_alloc_stats(bucket->site, bucket->size);
		}
	    }
	    p = &(bucket->next);
	}
    }
}

static void hprof_objmap_move(jobjectID obj_id, jint arena_id,
			      jobjectID new_obj_id, jint new_arena_id)
{
    
    int old_index = (unsigned long)obj_id % HPROF_OBJMAP_TABLE_SIZE;
    int new_index = (unsigned long)new_obj_id % HPROF_OBJMAP_TABLE_SIZE;
    hprof_objmap_t **p = hprof_objmap_table + old_index;
    hprof_objmap_t *bucket;
    
    /* remove it from the old position in the hash table */
    while ((bucket = *p)) {
        if (bucket->obj_id == obj_id) {
	    *p = bucket->next;
	    break;
	}
	p = &(bucket->next);
    }
    
    if (bucket != NULL) { 
        /* rehash using the new obj_id; NOTE - the bucket remains the same */
        bucket->obj_id = new_obj_id;
	bucket->arena_id = new_arena_id;
	bucket->next = hprof_objmap_table[new_index];
	hprof_objmap_table[new_index] = bucket;
    } /* we ignore moves for objects with no prior record */
}

hprof_objmap_t * hprof_objmap_lookup(jobjectID obj_id)
{
    int index = (unsigned long)obj_id % HPROF_OBJMAP_TABLE_SIZE;
    hprof_objmap_t *bucket = hprof_objmap_table[index];
    while (bucket) {
        if (bucket->obj_id == obj_id) {
	    return bucket;
	}
	bucket = bucket->next;
    }
    return NULL;
}

void hprof_objmap_print(hprof_objmap_t *objmap)
{
    hprof_class_t *hclass = objmap->site->class;
    hprof_printf(" ");
    switch(objmap->site->is_array) {
    case JVMPI_NORMAL_OBJECT:
        hprof_printf("%s", hclass ? hclass->name->name : "<unknown class>");
	break;
    case JVMPI_CLASS:
        hprof_printf("[L%s;", hclass->name->name);
	break;
    case JVMPI_BOOLEAN:
        hprof_printf("[Z");
	break;
    case JVMPI_BYTE:
        hprof_printf("[B");
	break;
    case JVMPI_CHAR:
        hprof_printf("[C");
	break;
    case JVMPI_SHORT:
        hprof_printf("[S");
	break;
    case JVMPI_INT:
        hprof_printf("[I");
	break;
    case JVMPI_LONG:
        hprof_printf("[J");
	break;
    case JVMPI_FLOAT:
        hprof_printf("[F");
	break;
    case JVMPI_DOUBLE:
        hprof_printf("[D");
	break;
    }
    hprof_printf("(%x)", objmap);
}

void hprof_obj_alloc_event(JNIEnv *env_id,
			   jobjectID class_id,
			   int is_array,
			   int size,
			   jobjectID obj_id,
			   jint arena_id,
			   int requested)
{ 
    hprof_site_t *site;
    int trace_num = 0;

    CALL(RawMonitorEnter)(data_access_lock);
    if (requested) {
        if (hprof_objmap_lookup(obj_id)) {
	    goto done;
	}
        trace_num = 0;
    } else {
        hprof_trace_t *htrace = hprof_get_trace(env_id, max_trace_depth);
	if (htrace == NULL) {
	    fprintf(stderr, "HPROF ERROR: got NULL trace in obj_alloc\n");
	} else {
	    trace_num = htrace->serial_num;
	}
    }
    site = hprof_intern_site(class_id, is_array, trace_num);
    hprof_objmap_add(obj_id, arena_id, site, size);
 done:
    CALL(RawMonitorExit)(data_access_lock);
}

void hprof_obj_free_event(JNIEnv *env_id,
			  jobjectID obj_id)
{
    /* data_access_lock already entered in GC_START event */
    hprof_objmap_del(obj_id);
}

void hprof_delete_arena_event(JNIEnv *env_id, 
			      jint arena_id)
{
    /* data_access_lock already entered in GC_START event */
    hprof_objmap_del_arena(arena_id);
}

void hprof_obj_move_event(JNIEnv *env_id,
			  jobjectID obj_id,
			  jint arena_id,
			  jobjectID new_obj_id,
			  jint new_arena_id)
{
    /* data_access_lock already entered in GC_START event */
    hprof_objmap_move(obj_id, arena_id, new_obj_id, new_arena_id);
}

hprof_objmap_t * hprof_fetch_object_info(jobjectID obj)
{
    hprof_objmap_t * objmap;
    if (obj == NULL) {
        return NULL;
    }
    objmap = hprof_objmap_lookup(obj);
    if (objmap == NULL) {
#ifdef OBJMAP_DEBUG
        fprintf(stderr, "requesting info for obj %x ", obj);
#endif
        CALL(RequestEvent)(JVMPI_EVENT_OBJ_ALLOC, obj);
	objmap = hprof_objmap_lookup(obj);
#ifdef OBJMAP_DEBUG
	fprintf(stderr, "=> %x\n", objmap);
#endif
    }
    return objmap;
}

void hprof_print_object_info(jobjectID obj)
{
    hprof_objmap_t *objmap = hprof_fetch_object_info(obj);
    hprof_class_t *hclass = objmap->site->class;
    if (objmap == NULL) {
        fprintf(stderr, "HPROF ERROR: unknown object ID 0x%p\n", obj);
    }
    hprof_printf(" ");
    switch(objmap->site->is_array) {
    case JVMPI_NORMAL_OBJECT:
        hprof_printf("%s", hclass ? hclass->name->name : "<unknown class>");
	break;
    case JVMPI_CLASS:
        hprof_printf("[L%s;", hclass->name->name);
	break;
    case JVMPI_BOOLEAN:
        hprof_printf("[Z");
	break;
    case JVMPI_BYTE:
        hprof_printf("[B");
	break;
    case JVMPI_CHAR:
        hprof_printf("[C");
	break;
    case JVMPI_SHORT:
        hprof_printf("[S");
	break;
    case JVMPI_INT:
        hprof_printf("[I");
	break;
    case JVMPI_LONG:
        hprof_printf("[J");
	break;
    case JVMPI_FLOAT:
        hprof_printf("[F");
	break;
    case JVMPI_DOUBLE:
        hprof_printf("[D");
	break;
    }
    hprof_printf("(%x)", objmap);
}
