/*
 * @(#)hprof.c	1.35 98/09/02
 *
 * Copyright 1997, 1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stddef.h>
#include <sys/types.h>

#include "hprof.h"

static void hprof_notify_event(JVMPI_Event *event);
static void hprof_dump_data(void);

JavaVM *jvm;

int hprof_is_on;            /* whether hprof is enabled */
int hprof_fd = -1;	    /* Non-zero file or socket descriptor. */
int hprof_socket_p = FALSE; /* True if hprof_fd is a socket. */

#define HPROF_DEFAULT_TRACE_DEPTH 4
int max_trace_depth = HPROF_DEFAULT_TRACE_DEPTH;
int prof_trace_depth = HPROF_DEFAULT_TRACE_DEPTH;
int thread_in_traces = FALSE;
int lineno_in_traces = TRUE;
char output_format = 'a';              /* 'b' for binary */
float hprof_cutoff_point = 0.0001;
int dump_on_exit = TRUE;
int cpu_sampling = FALSE;
int monitor_tracing = FALSE;
int heap_dump = FALSE;
int alloc_sites = FALSE;

/* cpu timing output formats */
int cpu_timing = FALSE;
int timing_format = NEW_PROF_OUTPUT_FORMAT;

jlong total_alloced_bytes;
jlong total_alloced_instances;
long total_live_bytes = 0;
long total_live_instances = 0;

/*
 * PRESERVE THIS LOCK ORDER TO AVOID DEADLOCKS!!!
 *
 * hprof_dump_lock => DisableGC/EnableGC => data_access_lock
 *
 */

/* Data access Lock */
JVMPI_RawMonitor data_access_lock;
/* Dump lock */
JVMPI_RawMonitor hprof_dump_lock;

/* profiler interface */
JVMPI_Interface *hprof_jvm_interface;

unsigned int micro_sec_ticks;

/**
 * Memory allocation and deallocation.
 */
void *hprof_calloc(unsigned int size)
{
    void *p = malloc(size);
    if (p == NULL) {
        fprintf(stderr, "HPROF ERROR: ***Out of Memory***, exiting..\n");
	CALL(ProfilerExit)((jint)1);
    }
    memset(p, 0, size);
    return p;
}

/* hprof initialisation */
JNIEXPORT jint JNICALL JVM_OnLoad(JavaVM *vm, char *options, void *reserved)
{
    int res;
    jvm = vm;
    res = (*jvm)->GetEnv(jvm, (void **)&hprof_jvm_interface, JVMPI_VERSION_1);
    if (res < 0) {
	return JNI_ERR;
    }
    hprof_jvm_interface->NotifyEvent = hprof_notify_event;
    hprof_dump_lock = CALL(RawMonitorCreate)("_hprof_dump_lock");
    hprof_init_setup(options);

    return JNI_OK;
}

static void hprof_reset_data(void)
{
    if (cpu_sampling || cpu_timing) {
        hprof_clear_trace_cost();
    }
    if (monitor_tracing) {
        hprof_clear_contended_monitor_table();
    }
}

static void hprof_dump_data(void)
{
    fprintf(stderr, "Dumping");
    if (monitor_tracing) {
        fprintf(stderr, " contended monitor usage ...");
        hprof_dump_monitors();
	hprof_output_cmon_times(hprof_cutoff_point);
    }
    if (heap_dump) {
        fprintf(stderr, " Java heap ...");
	hprof_get_heap_dump(); 
    }
    if (alloc_sites) {
        fprintf(stderr, " allocation sites ...");
	hprof_output_sites(0, hprof_cutoff_point);
    }
    if (cpu_sampling) {
        fprintf(stderr, " CPU usage by sampling running threads ...");
        hprof_output_trace_cost(hprof_cutoff_point, CPU_SAMPLES_RECORD_NAME);
    }
    if (cpu_timing) {
        hprof_bill_all_thread_local_tables();
	if (timing_format == NEW_PROF_OUTPUT_FORMAT) {
	    fprintf(stderr, " CPU usage by timing methods ...");
	    hprof_output_trace_cost(hprof_cutoff_point, CPU_TIMES_RECORD_NAME);
	} else if (timing_format == OLD_PROF_OUTPUT_FORMAT) {
	    fprintf(stderr, " CPU usage in old prof format ...");
	    hprof_output_trace_cost_in_prof_format();
	}
    }
    hprof_reset_data();
    hprof_flush();
    fprintf(stderr, " done.\n");
}


static void hprof_jvm_shut_down_event(void)
{
    static int already_dumped = FALSE;
    CALL(RawMonitorEnter)(hprof_dump_lock);
    if (!hprof_is_on || already_dumped) {
        CALL(RawMonitorExit)(hprof_dump_lock);
	return;
    }
    already_dumped = TRUE;
    if (dump_on_exit) {
        hprof_dump_data();
    }
    hprof_is_on = FALSE;
    close(hprof_fd);
    CALL(RawMonitorExit)(hprof_dump_lock);
}  

static void hprof_jvm_init_done_event(void)
{
    hprof_start_listener_thread();
    hprof_start_cpu_sampling_thread();
}

static void hprof_gc_start()
{
    CALL(RawMonitorEnter)(data_access_lock);
}

static void hprof_gc_finish(JVMPI_Event *event)
{
    CALL(RawMonitorExit)(data_access_lock);
}

static void hprof_notify_event(JVMPI_Event *event)
{
    switch(event->event_type) {
    case JVMPI_EVENT_JVM_INIT_DONE:
        hprof_jvm_init_done_event();
	return;
    case JVMPI_EVENT_JVM_SHUT_DOWN:
        hprof_jvm_shut_down_event();
	return;
    case JVMPI_EVENT_DUMP_DATA_REQUEST:
        CALL(RawMonitorEnter)(hprof_dump_lock);
        hprof_dump_data();
	CALL(RawMonitorExit)(hprof_dump_lock);
	return;
    case JVMPI_EVENT_RESET_DATA_REQUEST:
        hprof_reset_data();
	return;
    case JVMPI_EVENT_CLASS_LOAD: 
    case JVMPI_EVENT_CLASS_LOAD | JVMPI_REQUESTED_EVENT: 
        hprof_class_load_event(event->env_id, 
			       event->u.class_load.class_name,
			       event->u.class_load.source_name,
			       event->u.class_load.num_interfaces,
			       event->u.class_load.num_static_fields,
			       event->u.class_load.statics,
			       event->u.class_load.num_instance_fields,
			       event->u.class_load.instances,
			       event->u.class_load.num_methods,
			       event->u.class_load.methods,
			       event->u.class_load.class_id,
			       event->event_type & JVMPI_REQUESTED_EVENT);
	return;
    case JVMPI_EVENT_CLASS_UNLOAD:
        hprof_class_unload_event(event->env_id,
				 event->u.class_unload.class_id);
	return;
    case JVMPI_EVENT_OBJECT_DUMP | JVMPI_REQUESTED_EVENT:
        hprof_object_dump_event(event->u.object_dump.data);
	return;
    case JVMPI_EVENT_OBJ_ALLOC:
    case JVMPI_EVENT_OBJ_ALLOC | JVMPI_REQUESTED_EVENT:
        hprof_obj_alloc_event(event->env_id,
			      event->u.obj_alloc.class_id,
			      event->u.obj_alloc.is_array,
			      event->u.obj_alloc.size,
			      event->u.obj_alloc.obj_id,
			      event->u.obj_alloc.arena_id,
			      event->event_type & JVMPI_REQUESTED_EVENT);
	return;
    case JVMPI_EVENT_OBJ_FREE:
        hprof_obj_free_event(event->env_id,
			     event->u.obj_free.obj_id);
	return;
    case JVMPI_EVENT_OBJ_MOVE:
        hprof_obj_move_event(event->env_id,
			     event->u.obj_move.obj_id,
			     event->u.obj_move.arena_id,
			     event->u.obj_move.new_obj_id,
			     event->u.obj_move.new_arena_id);
	return;
    case JVMPI_EVENT_DELETE_ARENA:
        hprof_delete_arena_event(event->env_id,
				 event->u.delete_arena.arena_id);
	return;
    case JVMPI_EVENT_THREAD_START:
    case JVMPI_EVENT_THREAD_START | JVMPI_REQUESTED_EVENT:
        hprof_thread_start_event(event->u.thread_start.thread_env_id,
				 event->u.thread_start.thread_name,
				 event->u.thread_start.group_name,
				 event->u.thread_start.parent_name,
				 event->u.thread_start.thread_id,
				 event->event_type & JVMPI_REQUESTED_EVENT);
	return;
    case JVMPI_EVENT_THREAD_END:
        hprof_thread_end_event(event->env_id);
	return;
    case JVMPI_EVENT_METHOD_ENTRY:
        hprof_method_entry_event(event->env_id,
				 event->u.method.method_id);
	return;
    case JVMPI_EVENT_METHOD_EXIT:
        hprof_method_exit_event(event->env_id,
				event->u.method.method_id);
	return;
    case JVMPI_EVENT_HEAP_DUMP | JVMPI_REQUESTED_EVENT:
        hprof_heap_dump_event(event->u.heap_dump.begin,
			      event->u.heap_dump.end,
			      event->u.heap_dump.num_traces,
			      event->u.heap_dump.traces);
	return;
    case JVMPI_EVENT_JNI_GLOBALREF_ALLOC:
        hprof_jni_globalref_alloc_event(event->env_id,
					event->u.jni_globalref_alloc.obj_id,
					event->u.jni_globalref_alloc.ref_id);
	return;
    case JVMPI_EVENT_JNI_GLOBALREF_FREE:
        hprof_jni_globalref_free_event(event->env_id,
				       event->u.jni_globalref_free.ref_id);
	return;
    case JVMPI_EVENT_RAW_MONITOR_CONTENDED_ENTER:
    case JVMPI_EVENT_RAW_MONITOR_CONTENDED_ENTERED:
    case JVMPI_EVENT_RAW_MONITOR_CONTENDED_EXIT:
        hprof_raw_monitor_event(event,
				event->u.raw_monitor.name,
				event->u.raw_monitor.id);
	return;
    case JVMPI_EVENT_MONITOR_CONTENDED_ENTER:
    case JVMPI_EVENT_MONITOR_CONTENDED_ENTERED:
    case JVMPI_EVENT_MONITOR_CONTENDED_EXIT:
    case JVMPI_EVENT_MONITOR_WAITED:
    case JVMPI_EVENT_MONITOR_WAIT:
        hprof_monitor_event(event, event->u.monitor.object);
	return;
    case JVMPI_EVENT_MONITOR_DUMP | JVMPI_REQUESTED_EVENT:
        hprof_monitor_dump_event(event);
	return;
    case JVMPI_EVENT_GC_START:
        hprof_gc_start_event(event->env_id);
	return;
    case JVMPI_EVENT_GC_FINISH:
        hprof_gc_finish_event(event->env_id,
			      event->u.gc_info.used_objects,
			      event->u.gc_info.used_object_space,
			      event->u.gc_info.total_object_space);
	return;
    }

}
