#include <uhd/utils/thread_priority.hpp>
#include <uhd/utils/safe_main.hpp>
#include <uhd/usrp/multi_usrp.hpp>

#include <boost/program_options.hpp>
#include <boost/format.hpp>
#include <boost/thread.hpp>
#include <iostream>
#include <complex>
#include <stdio.h>
#include <cmath>

#include "single_rx.h"

namespace po = boost::program_options;

//System parameters 
double freq, gain, thres;
double inter;
double rate;

//USRP
uhd::stream_cmd_t stream_cmd(uhd::stream_cmd_t::STREAM_MODE_START_CONTINUOUS);

uhd::usrp::multi_usrp::sptr usrp1;
uhd::usrp::multi_usrp::sptr usrp2;
string usrp1_ip, usrp2_ip;

//TX/RX metadata
uhd::rx_metadata_t rx_md1, rx_md2, rx_md3;
//uhd::tx_metadata_t tx_md;

//Buffer
gr_complex *pkt1;
gr_complex *pkt2;


//File
string out1_name, out2_name; 

//Evaluation
size_t r_cnt;
double r_sec;
size_t s_cnt;

void init_usrp() {
	cout << "Initial USRP" << endl;
	
	// TODO: add usrp2
	usrp1 = uhd::usrp::multi_usrp::make(usrp1_ip);
	usrp1->set_rx_rate(rate);
	usrp1->set_tx_rate(rate);

	usrp1->set_rx_freq(freq);
	usrp1->set_tx_freq(freq);

	usrp1->set_rx_gain(gain);
	uhd::meta_range_t rx_range = usrp1->get_rx_gain_range();
}

void sync_clock() {
	cout << "SYNC Clock" << endl;
	// TODO: add usrp2
	usrp1->set_clock_config(uhd::clock_config_t::external());
	usrp1->set_time_next_pps(uhd::time_spec_t(0.0));
}

void init_stream() {
	// TODO: add usrp2
    boost::this_thread::sleep(boost::posix_time::milliseconds(WARM_UP_TIME));
    stream_cmd.time_spec = time_start_recv  = uhd::time_spec_t(2.0) + usrp1->get_time_now();
	cout << "Time to start receiving: " << time_start_recv.get_real_secs() << endl;
    stream_cmd.stream_now   = false;
    usrp1->issue_stream_cmd(stream_cmd);
}


void init_sys() {
	//Buffer initialize	
	printf("New receiving signal buffer\n");
	// TODO: receive two data streams
	s_cnt = (size_t)(1e8/SAMPLE_P * r_sec);
	pkt1 = new gr_complex[s_cnt];
	if(pkt1!= NULL) {
		memset(pkt1, 0, sizeof(gr_complex)*s_cnt);
	}

	rate = 1e8/inter;
	freq = 1e9*freq;

	init_usrp();
	sync_clock();
	init_stream();
}

void dump_signals() {
	cout << endl << "Dump signals" << endl;
	FILE *out1_file, *out2_file;
	char tmp_n[1000];
	// dump file for ant1
	sprintf(tmp_n, "%s", out1_name.c_str());
	out1_file = fopen(tmp_n, "wb");
	fwrite(pkt1, sizeof(gr_complex), s_cnt, out1_file );
	fclose(out1_file);
	// dump file for ant2
	sprintf(tmp_n, "%s", out2_name.c_str());
	out2_file = fopen(tmp_n, "wb");
	fwrite(pkt2, sizeof(gr_complex), s_cnt, out2_file );
	fclose(out2_file);
}

void end_sys() {
	printf("Delete receiving signal buffer\n");
	if(pkt1 != NULL) {
			delete [] pkt1;
	}
	if(pkt2 != NULL) {
			delete [] pkt2;
	}
}

int UHD_SAFE_MAIN(int argc, char *argv[]){
	size_t rx_cnt;
	rx_cnt = 0;

	uhd::set_thread_priority_safe();
	uhd::time_spec_t refer;

	po::options_description desc("Allowed options");
	desc.add_options()
		("help", "help message")
		("r0", po::value<string>(&usrp1_ip)->default_value("addr=192.168.20.2" ), "usrp's IP")
		("r1", po::value<string>(&usrp2_ip)->default_value("addr=192.168.10.2" ), "usrp's IP")
		("out1", po::value<string>(&out1_name)->default_value("wcs_trace/rx1_signal.bin"), "signal file")
		("out2", po::value<string>(&out2_name)->default_value("wcs_trace/rx2_signal.bin"), "signal file")
		("i", po::value<double>(&inter)->default_value(SAMPLE_P), "interval of two sampling")
		("f", po::value<double>(&freq)->default_value(2.49), "RF center frequency in Hz")
		("g", po::value<double>(&gain)->default_value(30.0), "gain for the RF chain")
		("s", po::value<double>(&r_sec)->default_value(RECV_SEC), "recording seconds")
		("c", po::value<size_t>(&r_cnt)->default_value(90), "round count");

	po::variables_map vm;
	po::store(po::parse_command_line(argc, argv, desc), vm);
	po::notify(vm);


	if (vm.count("help")){
		cout << boost::format("UHD TX samples from file %s") % desc << endl;
		return ~0;
	}
	// Initial systems
	init_sys();

	size_t cleaning, done_cleaning;
	done_cleaning = 0;
	cleaning = 0;
	while(cleaning < ANT_CNT) {
		if (!done_cleaning) {
			// TODO: add usrp2
			usrp1->get_device()->recv(pkt1, SYM_LEN, rx_md1, C_FLOAT32, R_ONE_PKT);
			if(rx_md1.time_spec.get_real_secs() >= time_start_recv.get_real_secs()) {
				done_cleaning = 1;
				cleaning++;
			}
		}
		// cout << cleaning << "-" << done_cleaning << " Clean ant" << i << " buff:" << rx_md1.time_spec.get_real_secs() << endl;
	}
	
	// remove content of within while loop
	cout << endl << "# of recv samples: " << s_cnt << endl;
	while(rx_cnt < s_cnt) {
		size_t  read_cnt = 80;
		//At last recv(), modify read_cnt to receive the remaining samples
		if (s_cnt - rx_cnt < read_cnt)
			read_cnt = s_cnt - rx_cnt;
		rx_cnt += usrp1->get_device()->recv(pkt1+rx_cnt, read_cnt, rx_md1, C_FLOAT32, R_ONE_PKT);

		if (rx_cnt < 100)
			cout << "Ant" << " recving at " << rx_md1.time_spec.get_real_secs() << endl;
	}
	

	stream_cmd.stream_mode = uhd::stream_cmd_t::STREAM_MODE_STOP_CONTINUOUS;
	usrp1->issue_stream_cmd(stream_cmd);
	
	boost::this_thread::sleep(boost::posix_time::seconds(1)); //allow for some setup time
	
	
	// End systems
	dump_signals();
	end_sys();	
	
	cout << "Terminate systems ... " << endl << endl;
	return 0;
}

