// =============================================================================
//  Program : pyramid.c
//  Author  : Chun-Jen Tsai
//  Date    : Sep/18/2019
// -----------------------------------------------------------------------------
//  Revision information:
//
//  None.
// -----------------------------------------------------------------------------
//  Description:
//
//  This is the C model of a 24-level image pyramid generator.
// -----------------------------------------------------------------------------
//  License information:
//
//  Copyright: MMES Lab.
//             Deparment of Computer Science
//             National Chiao Tung Uniersity
//             Hsinchu, Taiwan.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
//  Initial release: Sep. 2019
// =============================================================================
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

extern int xDownsampling3(unsigned char *, int, int, int, int, int);

unsigned char *read_pgm(char *fname, int *width, int *height)
{
    FILE *fp;
    char buf[32];
    unsigned char *image;

    if ((fp = fopen(fname, "rb")) == NULL)
    {
        fprintf(stderr, "read_pgm(): file '%s' open error.\n", fname);
        return NULL;
    }

    fgets(buf, sizeof(buf), fp); // read image color type
    fgets(buf, sizeof(buf), fp); // read image width & height
    if (sscanf(buf, "%d %d", width, height) != 2)
    {
        fprintf(stderr, "read_pgm(): image format error.\n");
        fclose(fp);
        return NULL;
    }
    fgets(buf, sizeof(buf), fp); // read depth per pixel per channel
    if ((image = (unsigned char*) malloc((*width)*(*height))) == NULL)
    {
        fprintf(stderr, "read_pgm(): out of memory.\n");
        fclose(fp);
        return NULL;
    }
    if (fread(image, *width, *height, fp) != *height) // read the image
    {
        fprintf(stderr, "read_pgm(): image data read error.\n");
        fclose(fp);
        return NULL;
    }
    fclose(fp);
    return image;
}

int write_pgm(char *fname, unsigned char *image, int width, int height)
{
    FILE *fp;

    if ((fp = fopen(fname, "wb")) == NULL)
    {
        fprintf(stderr, "write_pgm(): file '%s' open error.\n", fname);
        return 0;
    }

    fprintf(fp, "P5\n"); // image color type
    fprintf(fp, "%d %d\n", width, height); // image width & height
    fprintf(fp, "255\n");

    if (fwrite(image, width, height, fp) != height)
    {
        fprintf(stderr, "write_pgm(): file '%s' write error.\n", fname);
        fclose(fp);
        return 0;
    }

    fclose(fp);
    return 1;
}

int main(int argc, char **argv)
{
    int width, height, down_level, owidth, oheight;
    unsigned char *image, *ibuf;
    char ofname[64];

    if (argc < 2)
    {
        fprintf(stderr, "Usage: %s input.pgm\n\n", *argv);
        fprintf(stderr, "Generate a 24-level image pyramid. The image at each\n");
        fprintf(stderr, "level is saved as an independent 8-bit PGM file.\n");
        fprintf(stderr, "The output file names are ip01.pgm ~ ip24.pgm.\n");
        return 1;
    }

    // Read the input image
    image = read_pgm(argv[1], &width, &height);
    if (image == NULL)
    {
        fprintf(stderr, "%s: Down-scaling failed.\n", *argv);
        return 1;
    }

    // Allocate an image buffer
    if ((ibuf = (unsigned char*) malloc(width*height)) == NULL)
    {
        fprintf(stderr, "%s: out of memory.\n", *argv);
        return 1;
    }

    for (down_level = 31; down_level >= 8; down_level--)
    {
        memcpy(ibuf, image, width*height);
        owidth = (width * down_level) / 32;
        oheight =  (height * down_level) / 32;

        if (!xDownsampling3(ibuf, down_level, width, height, owidth, oheight))
        {
            printf("xDownsampling3(): memory allocation fails.\n");
            exit(0);
        }

        sprintf(ofname, "ip%02d.pgm", down_level - 7);
        write_pgm(ofname, ibuf, owidth, oheight);
    }

    free(image);
    free(ibuf);
    return 0;
}
