// =============================================================================
//  Program : dscale.c
//  Author  : Chun-Jen Tsai
//  Date    : Sep/19/2019
// -----------------------------------------------------------------------------
//  Revision information:
//
//  None.
// -----------------------------------------------------------------------------
//  Description:
//
//  This program reads an 8-bit grayscale PGM image and down scale it
//  by a decimation rate of 31/32, 30/32, ..., or 8/32. The Sine-Windowed
//  Sinc (SWS) function filters are used for the 2D down-scaling.
//
//  Usage: dscale down-level input.pgm output.pgm
//
// -----------------------------------------------------------------------------
//  License information:
//
//  Copyright: MMES Lab.
//             Deparment of Computer Science
//             National Chiao Tung Uniersity
//             Hsinchu, Taiwan.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
//  Initial release: Sep. 2019
// =============================================================================
#include <stdio.h>
#include <stdlib.h>

typedef unsigned char byte;
int xDownsampling3(byte *image, int level, int in_w, int in_h, int out_w, int out_h);

unsigned char *read_pgm(char *fname, int *width, int *height)
{
    FILE *fp;
    char buf[32];
    unsigned char *image;

    if ((fp = fopen(fname, "rb")) == NULL)
    {
        fprintf(stderr, "read_pgm(): file '%s' open error.\n", fname);
        return NULL;
    }

    fgets(buf, sizeof(buf), fp); // read image color type
    fgets(buf, sizeof(buf), fp); // read image width & height
    if (sscanf(buf, "%d %d", width, height) != 2)
    {
        fprintf(stderr, "read_pgm(): image format error.\n");
        fclose(fp);
        return NULL;
    }
    fgets(buf, sizeof(buf), fp); // read depth per pixel per channel
    if ((image = (unsigned char*) malloc((*width)*(*height))) == NULL)
    {
        fprintf(stderr, "read_pgm(): out of memory.\n");
        fclose(fp);
        return NULL;        
    }
    if (fread(image, *width, *height, fp) != *height) // read the image
    {
        fprintf(stderr, "read_pgm(): image data read error.\n");
        fclose(fp);
        return NULL;
    }
    fclose(fp);
    return image;
}

int write_pgm(char *fname, unsigned char *image, int width, int height)
{
    FILE *fp;

    if ((fp = fopen(fname, "wb")) == NULL)
    {
        fprintf(stderr, "write_pgm(): file '%s' open error.\n", fname);
        return 0;
    }

    fprintf(fp, "P5\n"); // image color type
    fprintf(fp, "%d %d\n", width, height); // image width & height
    fprintf(fp, "255\n");

    if (fwrite(image, width, height, fp) != height)
    {
        fprintf(stderr, "write_pgm(): file '%s' write error.\n", fname);
        fclose(fp);
        return 0;
    }

    fclose(fp);
    return 1;
}

int main(int argc, char **argv)
{
    int width, height, down_level, owidth, oheight;
    unsigned char *image;

    if (argc < 4)
    {
        fprintf(stderr, "Down-scale an 8-bit PGM image to level/32\n");
        fprintf(stderr, "Usage: %s level input.pgm output.pgm\n", *argv);
        fprintf(stderr, "Note: level is an integer from 8 ~ 31.\n");
        return 1;
    }
    if (sscanf(argv[1], "%d", &down_level) != 1)
    {
        fprintf(stderr, "%s: no level value.\n", *argv);
        return 1;
    }
    else if (down_level < 8 || down_level > 31)
    {
        fprintf(stderr, "%s: level value error.\n", *argv);
        return 1;
    }

    image = read_pgm(argv[2], &width, &height);
    if (image == NULL)
    {
        fprintf(stderr, "%s: Down-scaling failed.\n", *argv);
        return 1;
    }

    owidth = (width * down_level) / 32;
    oheight =  (height * down_level) / 32;

    printf("%dx%d -> %dx%d\n", width, height, owidth, oheight);

    if (!xDownsampling3(image, down_level, width, height, owidth, oheight))
    {
        printf("xDownsampling3: memory allocation fails.\n");
        exit(0);
    }

    write_pgm(argv[3], image, owidth, oheight);
    free(image);
    return 0;
}
