// =============================================================================
//  Program : down_scaler.c
//  Author  : Chiang-Yi Wang
//  Date    : 2017
// -----------------------------------------------------------------------------
//  Revision information:
//
//  Revised on Sep/18/2019 by Chun-Jen Tsai:
//    The original C model from
//    Chiang-Yi Wang was a single-piece program with lots of global variables
//    and embedded debugging code. I have rewritten the code to make it an
//    reusable function.
// -----------------------------------------------------------------------------
//  Description:
//
//  This is the C model of the Sine-Windowed Sinc (SWS) function filter
//  for an 8-bit grayscale image. The C model was written based on the
//  reference software from MPEG for the scalable extension of H.264/AVC.
//
//  This down scaler supports 24-level decimation rate, with the ranges:
//  31/32, 30/32, 29/32, ..., 8/32.  The input image must be a single-color
//  channel image.
//
// -----------------------------------------------------------------------------
//  License information:
//
//  Copyright: MMES Lab.
//             Deparment of Computer Science
//             National Chiao Tung Uniersity
//             Hsinchu, Taiwan.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//
//  Initial release: Sep. 2019
// =============================================================================
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#define XSIZE 640
#define YSIZE 480

typedef unsigned char byte;

// ====================================================================
//  Down-scaler
// ====================================================================

int xClip(int iValue, int imin, int imax)
{
    return (iValue < imin ? imin : iValue > imax ? imax : iValue);
}

const int filter16[8][16][12] =
{
    {                       // D = 1
            {0, 0, 0, 0, 0, 128, 0, 0, 0, 0, 0, 0},
            {0, 0, 0, 2, -6, 127, 7, -2, 0, 0, 0, 0},
            {0, 0, 0, 3, -12, 125, 16, -5, 1, 0, 0, 0},
            {0, 0, 0, 4, -16, 120, 26, -7, 1, 0, 0, 0},
            {0, 0, 0, 5, -18, 114, 36, -10, 1, 0, 0, 0},
            {0, 0, 0, 5, -20, 107, 46, -12, 2, 0, 0, 0},
            {0, 0, 0, 5, -21, 99, 57, -15, 3, 0, 0, 0},
            {0, 0, 0, 5, -20, 89, 68, -18, 4, 0, 0, 0},
            {0, 0, 0, 4, -19, 79, 79, -19, 4, 0, 0, 0},
            {0, 0, 0, 4, -18, 68, 89, -20, 5, 0, 0, 0},
            {0, 0, 0, 3, -15, 57, 99, -21, 5, 0, 0, 0},
            {0, 0, 0, 2, -12, 46, 107, -20, 5, 0, 0, 0},
            {0, 0, 0, 1, -10, 36, 114, -18, 5, 0, 0, 0},
            {0, 0, 0, 1, -7, 26, 120, -16, 4, 0, 0, 0},
            {0, 0, 0, 1, -5, 16, 125, -12, 3, 0, 0, 0},
            {0, 0, 0, 0, -2, 7, 127, -6, 2, 0, 0, 0}
    },
    {                       // D = 1.5
            {0, 2, 0, -14, 33, 86, 33, -14, 0, 2, 0, 0},
            {0, 1, 1, -14, 29, 85, 38, -13, -1, 2, 0, 0},
            {0, 1, 2, -14, 24, 84, 43, -12, -2, 2, 0, 0},
            {0, 1, 2, -13, 19, 83, 48, -11, -3, 2, 0, 0},
            {0, 0, 3, -13, 15, 81, 53, -10, -4, 3, 0, 0},
            {0, 0, 3, -12, 11, 79, 57, -8, -5, 3, 0, 0},
            {0, 0, 3, -11, 7, 76, 62, -5, -7, 3, 0, 0},
            {0, 0, 3, -10, 3, 73, 65, -2, -7, 3, 0, 0},
            {0, 0, 3, -9, 0, 70, 70, 0, -9, 3, 0, 0},
            {0, 0, 3, -7, -2, 65, 73, 3, -10, 3, 0, 0},
            {0, 0, 3, -7, -5, 62, 76, 7, -11, 3, 0, 0},
            {0, 0, 3, -5, -8, 57, 79, 11, -12, 3, 0, 0},
            {0, 0, 3, -4, -10, 53, 81, 15, -13, 3, 0, 0},
            {0, 0, 2, -3, -11, 48, 83, 19, -13, 2, 1, 0},
            {0, 0, 2, -2, -12, 43, 84, 24, -14, 2, 1, 0},
            {0, 0, 2, -1, -13, 38, 85, 29, -14, 1, 1, 0}
    },
    {                       // D = 2
            {2, 0, -10, 0, 40, 64, 40, 0, -10, 0, 2, 0},
            {2, 1, -9, -2, 37, 64, 42, 2, -10, -1, 2, 0},
            {2, 1, -9, -3, 34, 64, 44, 4, -10, -1, 2, 0},
            {2, 1, -8, -5, 31, 63, 47, 6, -10, -2, 3, 0},
            {1, 2, -8, -6, 29, 62, 49, 8, -10, -2, 3, 0},
            {1, 2, -7, -7, 26, 61, 52, 10, -10, -3, 3, 0},
            {1, 2, -6, -8, 23, 60, 54, 13, -10, -4, 3, 0},
            {1, 2, -6, -9, 20, 59, 56, 15, -10, -4, 3, 1},
            {1, 2, -5, -9, 18, 57, 57, 18, -9, -5, 2, 1},
            {1, 3, -4, -10, 15, 56, 59, 20, -9, -6, 2, 1},
            {0, 3, -4, -10, 13, 54, 60, 23, -8, -6, 2, 1},
            {0, 3, -3, -10, 10, 52, 61, 26, -7, -7, 2, 1},
            {0, 3, -2, -10, 8, 49, 62, 29, -6, -8, 2, 1},
            {0, 3, -2, -10, 6, 47, 63, 31, -5, -8, 1, 2},
            {0, 2, -1, -10, 4, 44, 64, 34, -3, -9, 1, 2},
            {0, 2, -1, -10, 2, 42, 64, 37, -2, -9, 1, 2}
    },
    {                       // D = 2.5
            {0, -4, -7, 11, 38, 52, 38, 11, -7, -4, 0, 0},
            {0, -4, -7, 9, 37, 51, 40, 13, -6, -7, 2, 0},
            {0, -3, -7, 8, 35, 51, 41, 14, -5, -7, 1, 0},
            {0, -2, -8, 6, 33, 51, 42, 16, -5, -7, 2, 0},
            {0, -2, -8, 5, 32, 50, 43, 18, -4, -8, 2, 0},
            {0, -2, -8, 4, 30, 50, 45, 19, -3, -8, 1, 0},
            {0, -1, -8, 2, 28, 49, 46, 21, -2, -8, 1, 0},
            {0, -1, -8, 1, 26, 49, 47, 23, -1, -8, 0, 0},
            {0, 0, -8, 0, 24, 48, 48, 24, 0, -8, 0, 0},
            {0, 0, -8, -1, 23, 47, 49, 26, 1, -8, -1, 0},
            {0, 1, -8, -2, 21, 46, 49, 28, 2, -8, -1, 0},
            {0, 1, -8, -3, 19, 45, 50, 30, 4, -8, -2, 0},
            {0, 2, -8, -4, 18, 43, 50, 32, 5, -8, -2, 0},
            {0, 2, -7, -5, 16, 42, 51, 33, 6, -8, -2, 0},
            {0, 1, -7, -5, 14, 41, 51, 35, 8, -7, -3, 0},
            {0, 2, -7, -6, 13, 40, 51, 37, 9, -7, -4, 0}
    },
    {                       // D = 3
            {-2, -7, 0, 17, 35, 43, 35, 17, 0, -7, -5, 2},
            {-2, -7, -1, 16, 34, 43, 36, 18, 1, -7, -5, 2},
            {-1, -7, -1, 14, 33, 43, 36, 19, 1, -6, -5, 2},
            {-1, -7, -2, 13, 32, 42, 37, 20, 3, -6, -5, 2},
            {0, -7, -3, 12, 31, 42, 38, 21, 3, -6, -5, 2},
            {0, -7, -3, 11, 30, 42, 39, 23, 4, -6, -6, 1},
            {0, -7, -4, 10, 29, 42, 40, 24, 5, -6, -6, 1},
            {1, -7, -4, 9, 27, 41, 40, 25, 6, -5, -6, 1},
            {1, -6, -5, 7, 26, 41, 41, 26, 7, -5, -6, 1},
            {1, -6, -5, 6, 25, 40, 41, 27, 9, -4, -7, 1},
            {1, -6, -6, 5, 24, 40, 42, 29, 10, -4, -7, 0},
            {1, -6, -6, 4, 23, 39, 42, 30, 11, -3, -7, 0},
            {2, -5, -6, 3, 21, 38, 42, 31, 12, -3, -7, 0},
            {2, -5, -6, 3, 20, 37, 42, 32, 13, -2, -7, -1},
            {2, -5, -6, 1, 19, 36, 43, 33, 14, -1, -7, -1},
            {2, -5, -7, 1, 18, 36, 43, 34, 16, -1, -7, -2}
    },
    {                       // D = 3.5
            {-6, -3, 5, 19, 31, 36, 31, 19, 5, -3, -6, 0},
            {-6, -4, 4, 18, 31, 37, 32, 20, 6, -3, -6, -1},
            {-6, -4, 4, 17, 30, 36, 33, 21, 7, -3, -6, -1},
            {-5, -5, 3, 16, 30, 36, 33, 22, 8, -2, -6, -2},
            {-5, -5, 2, 15, 29, 36, 34, 23, 9, -2, -6, -2},
            {-5, -5, 2, 15, 28, 36, 34, 24, 10, -2, -6, -3},
            {-4, -5, 1, 14, 27, 36, 35, 24, 10, -1, -6, -3},
            {-4, -5, 0, 13, 26, 35, 35, 25, 11, 0, -5, -3},
            {-4, -6, 0, 12, 26, 36, 36, 26, 12, 0, -6, -4},
            {-3, -5, 0, 11, 25, 35, 35, 26, 13, 0, -5, -4},
            {-3, -6, -1, 10, 24, 35, 36, 27, 14, 1, -5, -4},
            {-3, -6, -2, 10, 24, 34, 36, 28, 15, 2, -5, -5},
            {-2, -6, -2, 9, 23, 34, 36, 29, 15, 2, -5, -5},
            {-2, -6, -2, 8, 22, 33, 36, 30, 16, 3, -5, -5},
            {-1, -6, -3, 7, 21, 33, 36, 30, 17, 4, -4, -6},
            {-1, -6, -3, 6, 20, 32, 37, 31, 18, 4, -4, -6}
    },
    {                       // D = 4
            {-9, 0, 9, 20, 28, 32, 28, 20, 9, 0, -9, 0},
            {-9, 0, 8, 19, 28, 32, 29, 20, 10, 0, -4, -5},
            {-9, -1, 8, 18, 28, 32, 29, 21, 10, 1, -4, -5},
            {-9, -1, 7, 18, 27, 32, 30, 22, 11, 1, -4, -6},
            {-8, -2, 6, 17, 27, 32, 30, 22, 12, 2, -4, -6},
            {-8, -2, 6, 16, 26, 32, 31, 23, 12, 2, -4, -6},
            {-8, -2, 5, 16, 26, 31, 31, 23, 13, 3, -3, -7},
            {-8, -3, 5, 15, 25, 31, 31, 24, 14, 4, -3, -7},
            {-7, -3, 4, 14, 25, 31, 31, 25, 14, 4, -3, -7},
            {-7, -3, 4, 14, 24, 31, 31, 25, 15, 5, -3, -8},
            {-7, -3, 3, 13, 23, 31, 31, 26, 16, 5, -2, -8},
            {-6, -4, 2, 12, 23, 31, 32, 26, 16, 6, -2, -8},
            {-6, -4, 2, 12, 22, 30, 32, 27, 17, 6, -2, -8},
            {-6, -4, 1, 11, 22, 30, 32, 27, 18, 7, -1, -9},
            {-5, -4, 1, 10, 21, 29, 32, 28, 18, 8, -1, -9},
            {-5, -4, 0, 10, 20, 29, 32, 28, 19, 8, 0, -9}
    },
    {                       // D = 5.5
            {-8, 7, 13, 18, 22, 24, 22, 18, 13, 7, 2, -10},
            {-8, 7, 13, 18, 22, 23, 22, 19, 13, 7, 2, -10},
            {-8, 6, 12, 18, 22, 23, 22, 19, 14, 8, 2, -10},
            {-9, 6, 12, 17, 22, 23, 23, 19, 14, 8, 3, -10},
            {-9, 6, 12, 17, 21, 23, 23, 19, 14, 9, 3, -10},
            {-9, 5, 11, 17, 21, 23, 23, 20, 15, 9, 3, -10},
            {-9, 5, 11, 16, 21, 23, 23, 20, 15, 9, 4, -10},
            {-9, 5, 10, 16, 21, 23, 23, 20, 15, 10, 4, -10},
            {-10, 5, 10, 16, 20, 23, 23, 20, 16, 10, 5, -10},
            {-10, 4, 10, 15, 20, 23, 23, 21, 16, 10, 5, -9},
            {-10, 4, 9, 15, 20, 23, 23, 21, 16, 11, 5, -9},
            {-10, 3, 9, 15, 20, 23, 23, 21, 17, 11, 5, -9},
            {-10, 3, 9, 14, 19, 23, 23, 21, 17, 12, 6, -9},
            {-10, 3, 8, 14, 19, 23, 23, 22, 17, 12, 6, -9},
            {-10, 2, 8, 14, 19, 22, 23, 22, 18, 12, 6, -8},
            {-10, 2, 7, 13, 19, 22, 23, 22, 18, 13, 7, -8}
    }
};

int xDownsampling3(byte *image, int down_level,
    int input_width, int input_height,
    int output_width, int output_height)
{
    int i, j, x, y, filter;
    int im6, im5, im4, im3, im2, im1, ip1, ip2, ip3, ip4, ip5, ip6;
    int clipped_value, size;
    int filter_num;

    int *int_image;   // integer-value image array
    int *output_buf;  // filtered integer-valued pixels
    int *fpid;        // filter number & pixel ID

    // Allocate internal working memory.
    size = input_width*input_height;
    if ((int_image = (int *) malloc(size*sizeof(int))) == NULL)
    {
        return 0;
    }
    size = (input_width > input_height)? input_width : input_height;
    if ((output_buf = (int *) malloc(size*sizeof(int))) == NULL)
    {
        return 0;
    }
    if ((fpid = (int *) malloc(size*sizeof(int))) == NULL)
    {
        return 0;
    }

    // Copy the 8-bit input image pixels to an integer array.
    for (i = 0; i < input_width*input_height; i++)
    {
        int_image[i] = (int) image[i];
    }

    //========== horizontal downsampling ===========
    if      (input_width *  4 > 15 * output_width) filter = 7;
    else if (input_width *  7 > 20 * output_width) filter = 6;
    else if (input_width *  2 >  5 * output_width) filter = 5;
    else if (input_width *  1 >  2 * output_width) filter = 4;
    else if (input_width *  3 >  5 * output_width) filter = 3;
    else if (input_width *  4 >  5 * output_width) filter = 2;
    else if (input_width * 19 > 20 * output_width) filter = 1;
    else                                           filter = 0;

    for (i = 0; i < output_width; i++)
    {
        fpid[i] = (i * (input_width << 4) + ((input_width << 3) -
                ((input_width * down_level) >> 2)) +
                ((input_width * down_level) >> 6)) /
                ((input_width * down_level) >> 5);
    }
    for (j = 0; j < input_height; j++)
    {
        int *piSrc = &int_image[j * input_width];
        for (i = 0; i < output_width; i++)
        {
            filter_num = fpid[i] & 0x0f;
            x = fpid[i] >> 4;

            im6 = (x < 6) ? 0 : x - 6;
            im5 = (x < 5) ? 0 : x - 5;
            im4 = (x < 4) ? 0 : x - 4;
            im3 = (x < 3) ? 0 : x - 3;
            im2 = (x < 2) ? 0 : x - 2;
            im1 = (x < 1) ? 0 : x - 1;

            ip1 = (x < input_width) ? x : input_width - 1;
            ip2 = (x < input_width - 1) ? x + 1 : input_width - 1;
            ip3 = (x < input_width - 2) ? x + 2 : input_width - 1;
            ip4 = (x < input_width - 3) ? x + 3 : input_width - 1;
            ip5 = (x < input_width - 4) ? x + 4 : input_width - 1;
            ip6 = (x < input_width - 5) ? x + 5 : input_width - 1;

            clipped_value =
                 (filter16[filter][filter_num][ 0] * piSrc[im6]
                + filter16[filter][filter_num][ 1] * piSrc[im5]
                + filter16[filter][filter_num][ 2] * piSrc[im4]
                + filter16[filter][filter_num][ 3] * piSrc[im3]
                + filter16[filter][filter_num][ 4] * piSrc[im2]
                + filter16[filter][filter_num][ 5] * piSrc[im1]
                + filter16[filter][filter_num][ 6] * piSrc[ip1]
                + filter16[filter][filter_num][ 7] * piSrc[ip2]
                + filter16[filter][filter_num][ 8] * piSrc[ip3]
                + filter16[filter][filter_num][ 9] * piSrc[ip4]
                + filter16[filter][filter_num][10] * piSrc[ip5]
                + filter16[filter][filter_num][11] * piSrc[ip6]) >> 7;

            output_buf[i] = xClip(clipped_value, 0, 255);
        }

        //----- copy row back to image buffer -----
        memcpy(piSrc, output_buf, output_width * sizeof(int));
    }

    //========== vertical downsampling ===========
    if      (input_height *  4 > 15 * output_height) filter = 7;
    else if (input_height *  7 > 20 * output_height) filter = 6;
    else if (input_height *  2 >  5 * output_height) filter = 5;
    else if (input_height *  1 >  2 * output_height) filter = 4;
    else if (input_height *  3 >  5 * output_height) filter = 3;
    else if (input_height *  4 >  5 * output_height) filter = 2;
    else if (input_height * 19 > 20 * output_height) filter = 1;
    else                                             filter = 0;

    for (j = 0; j < output_height; j++)
    {
        fpid[j] = (j * (input_height << 4) + ((input_height << 3) -
                ((input_height * down_level) >> 2)) +
                ((input_height * down_level) >> 6)) /
                ((input_height * down_level) >> 5);
    }
    for (i = 0; i < output_width; i++)
    {
        int *piSrc = &int_image[i];
        for (j = 0; j < output_height; j++)
        {
            filter_num = fpid[j] & 0x0f;
            y = fpid[j] >> 4;
            clipped_value = 0;

            im6 = ((y < 6) ? 0 : y - 6) * input_width;
            im5 = ((y < 5) ? 0 : y - 5) * input_width;
            im4 = ((y < 4) ? 0 : y - 4) * input_width;
            im3 = ((y < 3) ? 0 : y - 3) * input_width;
            im2 = ((y < 2) ? 0 : y - 2) * input_width;
            im1 = ((y < 1) ? 0 : y - 1) * input_width;

            ip1 = ((y < input_height) ? y : input_height - 1) * input_width;
            ip2 = ((y < input_height - 1) ? y + 1 : input_height - 1) * input_width;
            ip3 = ((y < input_height - 2) ? y + 2 : input_height - 1) * input_width;
            ip4 = ((y < input_height - 3) ? y + 3 : input_height - 1) * input_width;
            ip5 = ((y < input_height - 4) ? y + 4 : input_height - 1) * input_width;
            ip6 = ((y < input_height - 5) ? y + 5 : input_height - 1) * input_width;

            clipped_value =
                ( filter16[filter][filter_num][ 0] * piSrc[im6]
                + filter16[filter][filter_num][ 1] * piSrc[im5]
                + filter16[filter][filter_num][ 2] * piSrc[im4]
                + filter16[filter][filter_num][ 3] * piSrc[im3]
                + filter16[filter][filter_num][ 4] * piSrc[im2]
                + filter16[filter][filter_num][ 5] * piSrc[im1]
                + filter16[filter][filter_num][ 6] * piSrc[ip1]
                + filter16[filter][filter_num][ 7] * piSrc[ip2]
                + filter16[filter][filter_num][ 8] * piSrc[ip3]
                + filter16[filter][filter_num][ 9] * piSrc[ip4]
                + filter16[filter][filter_num][10] * piSrc[ip5]
                + filter16[filter][filter_num][11] * piSrc[ip6]) >> 7;

            output_buf[j] = xClip(clipped_value, 0, 255);
        }

        //----- scale and copy back to image buffer -----
        for (j = 0; j < output_height; j++)
        {
            image[j * output_width + i] = (unsigned char) xClip(output_buf[j], 0, 255);
        }
    }

    // Free allocated memory blocks.
    free(int_image);
    free(output_buf);
    free(fpid);
    return 1;
}
