package dice_war;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * A console server, multi-server model which use multi-thread to handle connections. 
 * The core role of this project.
 */
public class Server implements Runnable
{
	public static void main(String[] args)
	{
		Server server = new Server();
		server.startUp();
	}

	/********************************************************/
	boolean bRun = false;
	ServerSocket serverSocket;
	int port = 2046;
	
	int turn;
	int maxPlayer = 2;
	// map's parameters
	int width = 12, height = 8;
	int densityOfEdge = 70;
	int edgeLevel = 2;

	static final int ARRANGE_BY_WALL = 0;
	static final int ARRANGE_BY_UNIFORM = 1;
	static final int ARRANGE_BY_USER = 2;
	int arrangeNewDiceWay = ARRANGE_BY_WALL; 
	/*
	 * shared field, need be attention to synchrounization
	 */
	int nPlayer;
	Socket clientSocket;
	List agents;
	
	Map map;
	
	Server()
	{
		this(2046);
	}

	Server(int port)
	{
		readConfig();
		this.port = port;	  
		map = new Map(width, height, maxPlayer, edgeLevel, densityOfEdge);
		initData();
	}
	// used by ServerGUI which builed the map first and then pass it to Server
	Server(int port, int arrangeNewDiceWay, Map map)
	{
		this.port = port;
		this.arrangeNewDiceWay = arrangeNewDiceWay;
		this.map = map;
		maxPlayer = map.maxConnectedCount.length - 1;
		initData();
	}

	private void initData()
	{
		nPlayer = 0;
		agents = new ArrayList();
		try {
			serverSocket = new ServerSocket(port);
		} catch (Exception e) {
			System.out.println("Could not listen on port " + port + ": " + e);
			System.exit(-1);
		}
		map.generateMap();
	}
  
	Acceptor ac;
	void startUp()
	{
		ac = new Acceptor();
		Thread th = new Thread(ac);
		th.start();
	}
	
	private void readConfig()
	{
		String configFile = "dice_war/config.txt";
		try {
			FileReader fr = new FileReader(configFile);
			BufferedReader br = new BufferedReader(fr);
			String s, ss[], name, value;
			while ((s=br.readLine()) != null) {
				if (s.indexOf('#') >= 0)
					continue;
				ss = s.split(" ");
				if (ss.length != 2)
					continue;
				name = ss[0];
				value = ss[1];
				if (name.toLowerCase().equals("width")) {
					width = Integer.parseInt(value);
				} else if (name.toLowerCase().equals("height")) {
					height = Integer.parseInt(value);
				} else if (name.toLowerCase().equals("max_player")) {
					maxPlayer = Integer.parseInt(value);
				} else if (name.toLowerCase().equals("density_of_edge")) {
					densityOfEdge = Integer.parseInt(value);
				} else if (name.toLowerCase().equals("edge_level")) {
					edgeLevel = Integer.parseInt(value);
				} else if (name.toLowerCase().equals("arrange_new_dice_way")) {
					arrangeNewDiceWay = Integer.parseInt(value);; 
				}
			}
		} catch (FileNotFoundException fe) {
			System.out.println("[Warnning] cannot find config file: " + configFile);
		} catch (Exception e) {
			e.printStackTrace();
		}
		
	}

//	boolean waitClient;

	/**
	 * The receiver of client, connected with players.
	 * There is only one client is allowed to send messages, un-granted messages sent by other clients will be discarded 
	 */
	public void run()
	{
		Agent me = null;
		synchronized (this) { // clientSocket, agents, nPlayer
			try {
				if (nPlayer >= maxPlayer) {
					clientSocket.close();
					return;
				}
				
				nPlayer++;
				me = new Agent(clientSocket, nPlayer);
				agents.add(me);
				
				Message msg = new Message("set country", new Integer(nPlayer));
				me.sendMessage(msg); 
				msg = new Message("set map", map);
				me.sendMessage(msg);
				msg = new Message("set turn", new Integer(turn));
				me.sendMessage(msg); 
			} catch (Exception e2) {
				e2.printStackTrace();
				return;
			}
//			notifyAll();
		}
		
		Message msg = null;
		while (me.getSocket().isClosed() == false) {
			msg = me.readMessage();
			if (msg == null)
				break;
			if (turn != me.country) // not my turn, discard the msg to prevent cheating 
				continue;
			
			try {
				executeCommand(me, msg);
			} catch (ClassCastException ce) {
				System.out.println("execute command error: ");
				ce.printStackTrace();
			} catch (Exception e3) {
				System.out.println("execute command error: ");
				e3.printStackTrace();
				break;
			}
		}
		/******************* end **************************/
		synchronized (agents) {
			agents.remove(me);
		}
		
		try {
			if (me.getSocket().isClosed() == false) {
				me.getSocket().close();
				System.out.println("normal exit");
			}
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		
		nPlayer--;
		if (nPlayer == 0)
			stopServer();
		System.out.println("client exits ( from " + me.getSocket().getRemoteSocketAddress() + " ) ");
	}

	private void stopServer()
	{		
		if (serverSocket != null) {
			try {
				serverSocket.close();
			} catch (IOException e1) {
				e1.printStackTrace();
			}
			serverSocket = null;
		}
		bRun = false;
	}
	
	void broadcast(Message m)
	{
		for (int i=0; i<agents.size(); i++) {
			Agent a = (Agent)agents.get(i);
			a.sendMessage(m);
		}
	}
	
	int nNewDice;
	ServerGUI serverGUI;
	boolean bSetNNewDice;
//	boolean bEndTurn = false;
	// this method will call by only one thread, since other agents' message would be discarded
	boolean executeCommand(Agent a, Message msg) throws Exception
	{   	
		if (msg.cmd.equals("attack")) {
			int t[] = (int[])msg.arg;
			Node[] result = map.attack(t[0], t[1]);
			// TODO send the interm-process to let the clients play it
			Message m = new Message("update map", result);
			broadcast(m);
			doCountConnectedNode();
		} else if (msg.cmd.equals("end turn")) {
			Message m;
			
			doCountConnectedNode();
			
			int ruler = map.isAllBelongsToOne();
			if (ruler > 0) {
				m = new Message("game over", new Integer(ruler));
				broadcast(m);
//			} else if (bEndTurn == false) {
			} else {
//				bEndTurn = true;
				if (arrangeNewDiceWay == ARRANGE_BY_USER) {
//					bSetNNewDice = false; // debug
					if (bSetNNewDice == false) {
						bSetNNewDice = true;
						nNewDice = map.maxConnectedCount[turn];
						m = new Message("arrange new dices", new Integer(nNewDice));
						a.sendMessage(m);
					}
				} else {
					// auto arrangement by server
					Node[] un = null;
					if (arrangeNewDiceWay == ARRANGE_BY_WALL)
						un = map.arrangeByWall(turn);
					else if (arrangeNewDiceWay == ARRANGE_BY_UNIFORM)
						un = map.arrangeByUniformDistribution(turn);
					m = new Message("update map", un);
					broadcast(m);
					setNewTurn();
				}
			}
		} else if (msg.cmd.equals("append dice")) {
//			System.out.println("nNewDice = " + nNewDice);
			if (nNewDice > 0) {
				int n = ((Integer)msg.arg).intValue();			
				if (map.nodes[n].number < Map.MAX_DICE) {
					map.nodes[n].number++;
					Message m = new Message("update map", new Node[] { new Node(map.nodes[n]) });
					broadcast(m);
				}
				nNewDice--;
				if (nNewDice == 0) {
					bSetNNewDice = false;
					Message m = new Message("end arrangement", null);
					a.sendMessage(m);
					setNewTurn();				
				}
			}
		} else if (msg.cmd.equals("end my arrangement")) {
			Node[] un = map.arrangeByUniformDistribution(turn, nNewDice);
			Message m = new Message("update map", un);
			broadcast(m);
			
			nNewDice = 0;
			bSetNNewDice = false;
			m = new Message("end arrangement", null);
			a.sendMessage(m);
			setNewTurn();	
		} else {
			System.out.println("unkown message: " + msg);
		}
		//OUCH dirty code
		if (serverGUI != null)
			serverGUI.repaint();
		return true;
	}

	private void setNewTurn()
	{
		do {
			if (++turn > maxPlayer)
				turn = 1;
		} while (map.maxConnectedCount[turn] == 0);
		Message m = new Message("set turn", new Integer(turn));
		broadcast(m);
	}

	private void doCountConnectedNode()
	{
		Message m;
		map.countConnectedNodes();
		int[] ccn = copyIntArray(map.maxConnectedCount);			
		m = new Message("update count of connected nodes", ccn);
		broadcast(m);
	}

	private int[] copyIntArray(int[] a)
	{
		int[] na = new int[a.length];
		for (int i=0; i<na.length; i++)
			na[i] = a[i];
		return na;
	}

	private Server getServer() { return this; }

	/**
	 * listen in the port and wait for new connection
	 */
	class Acceptor implements Runnable
	{
		// wait for accepting for new connection
		public void run()
		{
			System.out.println("Server start up");
			bRun = true;
			turn = 1;
			while (bRun) {
				try {
					// TODO clientSocket may have sync problem?
					clientSocket = serverSocket.accept();
					// new connection, create a thread to handle it
					Thread th = new Thread(getServer(), "Dice War Server");
					th.start();
					
					System.out.println("new client from " + clientSocket.getRemoteSocketAddress());
				} catch (Exception e) {
					System.out.println("Accept failed: " + port + ": " + e);
					//break;
				}
			}
			stopServer();
		}
	}
	// used by ServerGUI
	void stop()
	{
		bRun = false;
		try {
			if (serverSocket != null)
				serverSocket.close();
			Thread.sleep(1000);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
}
