package dice_war;

import java.util.*;
import java.io.*;

/**
 * 
 * the core data structure represented the graph of the map
 *
 */
public class Map implements Serializable
{
	private static final long serialVersionUID = 6139197187062142568L;

	static final int MAX_DICE = 8;
	// parameters of generating map
	private int width = 8, height = 6;
	private int nCountry = 2;
	private int emptyLevel = 2;
	private int densityOfEdge = 70;
	Random random;
	
	private int nNode;
	
	Node[] nodes;
	boolean edge[][];
	int edgeList[][];
	
	// info
	// save the max size of connected subgraph for each country
	int maxConnectedCount[];
	
	/**
	 * construct a width x height map with some hole inside it
	 * 
	 * @param width the width of the map, valid range: [4,12]
	 * @param height the height of the map, valid range: [3,8]
	 * @param nCountry the number of players, valid range: [2,8]
	 * @param emptyLevel the larger value it is, the more holes there are, valid range: [0,20]
	 * @param densityOfEdge the larger value it is, the more edges there are, valid range: [0,100]
	 */
	Map(int width, int height, int nCountry, int emptyLevel, int densityOfEdge)
	{
		setWidth(width);
		setHeight(height);
		setNCountry(nCountry);
		setEmptyLevel(emptyLevel);
		setDensityOfEdge(densityOfEdge);
		random = new Random(Calendar.getInstance().getTimeInMillis());
	}

	// use for generating map
	boolean bVisited[];
	int to[];
	
	private void swap(int a[], int i, int j)
	{
		int t = a[i];
		a[i] = a[j];
		a[j] = t;
	}
	
	private void randomWalk(int n)
	{
		bVisited[n] = true;
		// decide the walk order
		int next[] = new int[] {
				-width, -width-1, -1, +width-1, 
				+width, +width+1, +1, -width+1
		};
		for (int i=0; i<next.length; i++) {
			int fr = random.nextInt(next.length);
			int to = random.nextInt(next.length);
			swap(next, fr, to);
		}
		for (int i=0; i<next.length; i++) {
			int k = n + next[i];
			if (k < 0 || k >= nodes.length) // || nodes[k].country == 0)
				continue;
			if (Math.abs(nodes[n].x-nodes[k].x) > 1 || Math.abs(nodes[n].y-nodes[k].y) > 1) 
				continue;
			if (bVisited[k] == false) {
				edge[n][k] = edge[k][n] = true;
				randomWalk(k);
			}
		}
	}
	/**
	 * generate a new map
	 *
	 */
	void generateMap()
	{
		// step 0: initilize data structures
		nodes = new Node[width*height];
		edge = new boolean[nodes.length][nodes.length];
		maxConnectedCount = new int[nCountry+1];
				
		for (int i=0; i<edge.length; i++)
			for (int j=0; j<edge[i].length; j++)
				edge[i][j] = false;
		
		for (int i=0; i<nodes.length; i++) {			
			int x = i % width + 1;
			int y = i / width + 1;
			// TODO change back
			int number = random.nextInt(MAX_DICE/2) + 1;
//			int number = random.nextInt(MAX_DICE) + 1;
//			int country = random.nextInt(nCountry) + 1;
			int country;
			do {
				country = (int)Math.ceil(random.nextGaussian() * nCountry) + 1;
			} while (country <= 0 || country > nCountry);
			nodes[i] = new Node(x, y, i, country, number);
		}
		to = new int[] { -width, -width-1, -1, width-1 };
		// step 1: random 2walk, build a tree (so the graph is connected)
		bVisited = new boolean[nodes.length];
		for (int i=0; i<nodes.length; i++)
			bVisited[i] = false;
		int root = nodes.length-1;
		do {
			root = random.nextInt(nodes.length);
		} while (nodes[root].x == 1 || nodes[root].y == 1 
						|| nodes[root].x == width || nodes[root].y == height);
		randomWalk(root);
		// step 2: set some leaves as empty node
		nNode = width * height;
		for (int k=0; k<emptyLevel; k++) {
			buildEdgeList(); // rebuild
			for (int i=0; i<nodes.length; i++) {
				if (edgeList[i].length == 1) { // leaf
					nodes[i].country = 0;
					nNode--;
					for (int j=0; j<edge[i].length; j++)
						edge[i][j] = edge[j][i] = false;
				}
			}	
		}
		// step 3: build the graph (add new edges) and rebuild edge list
		for (int i=0; i<nodes.length; i++) {
			for (int j=0; j<to.length; j++) {
				if ((nodes[i].x == 1 && j >= 1) || nodes[i].country == 0)
					continue;
				int k = i+to[j];
				if (k >= 0 && k < nodes.length && edge[i][k] == false)
					edge[i][k] = edge[k][i] = (random.nextInt(100)<=densityOfEdge ? true : false);
			}
		}
		// step 4: build the final edge list
		buildEdgeList();
		// step 5: count the connected nodes
		countConnectedNodes();
		// step 6: renamed the countries, 
		// so that the first country has the least connected nodes and the last one has the most
		for (int i=1; i<maxConnectedCount.length; i++) {
			int min = i;
			for (int j=i+1; j<maxConnectedCount.length; j++) {
				if (maxConnectedCount[j] < maxConnectedCount[min])
					min = j;
			}
			if (min == i)
				continue;
			// swap (min, i)
//			System.out.println("swap " + i + " " + min);
			int t = maxConnectedCount[min];
			maxConnectedCount[min] = maxConnectedCount[i];
			maxConnectedCount[i] = t;
			for (int k=0; k<nodes.length; k++) {
				if (nodes[k].country == i)
					nodes[k].country = min;
				else if (nodes[k].country == min)
					nodes[k].country = i;
			}				
		}
		System.out.println("map has been generated!");
	}

	private void buildEdgeList()
	{
		edgeList = new int[nodes.length][];
		for (int i=0; i<nodes.length; i++) {
			int count = 0;
			for (int j=0; j<edge[i].length; j++)
				if (edge[i][j])
					count++;
			edgeList[i] = new int[count];
			int k = 0;
			for (int j=0; j<edge[i].length; j++)
				if (edge[i][j])
					edgeList[i][k++] = j;
		}
	}

	// count the connected nodes
	private int countConnectedNode(int n)
	{
		if (bVisited[n])
			return 0;
		bVisited[n] = true;
		
		int count = 1;
		for (int i=0; i<edgeList[n].length; i++) {
			int next = edgeList[n][i];
			if (nodes[next].country == nodes[n].country)
				count += countConnectedNode(next);
		}
		return count;
	}
	
	void countConnectedNodes()
	{
		for (int i=0; i<maxConnectedCount.length; i++)
			maxConnectedCount[i] = 0;
		for (int i=0; i<nodes.length; i++)
			bVisited[i] = false;
		for (int i=0; i<nodes.length; i++) {
			int c = countConnectedNode(i);
			if (c > maxConnectedCount[nodes[i].country])
				maxConnectedCount[nodes[i].country] = c;
		}
	}
	
	// append the new dices on some nodes and return the copy of those updated node
	Node[] arrangeNewNodes(int country)
	{
//		return arrangeByUniformDistribution(country);
		return arrangeByWall(country);
	}
	// this method used by UserInterface when the user click the button "End Arrangement"
	Node[] arrangeByUniformDistribution(int country, int nArrangement)
	{
		ArrayList al = new ArrayList();
		// build up index structure
		ArrayList lists = new ArrayList();
		for (int i=0; i<nodes.length; i++)
			if (nodes[i].country == country)
				lists.add(nodes[i]);
		// make arrangement
		for (int i=0; i<nArrangement && lists.size()>0; ) {
			int k = random.nextInt(lists.size());
			Node n = (Node)lists.get(k);
			if (n.number < MAX_DICE) {
				n.number++;
				i++;
				al.add(n);
			} else {
				lists.remove(k);
			}
		}
		// return update nodes
		return extractUpdatedNodes(al);
	}
	
	Node[] arrangeByUniformDistribution(int country)
	{
		return arrangeByUniformDistribution(country, maxConnectedCount[country]);
	}
		
	Node[] arrangeByWall(int country)
	{
		ArrayList al = new ArrayList();
		// build up index structure
		ArrayList wall = new ArrayList();
		ArrayList lists = new ArrayList();
		for (int i=0; i<nodes.length; i++)
			if (nodes[i].country == country) {
				int j;
				for (j=0; j<edgeList[i].length; j++) {
					int neighbor = edgeList[i][j];
					if (nodes[neighbor].country != 0 && nodes[neighbor].country != country) {
						// the more enemy, the more change to get new dices
						wall.add(nodes[i]);
					}
				}
				lists.add(nodes[i]);
			}
		// make arrangement
		// wall nodes take half of new nodes first
		int i;
		int nArrangement = maxConnectedCount[country];
		for (i=0; i<nArrangement/2 && wall.size()>0; ) {
			int k = random.nextInt(wall.size());
			Node n = (Node)wall.get(k);
			if (n.number < MAX_DICE) {
				n.number++;
				i++;
				al.add(n);
			} else {
				wall.remove(k);
			}
		}
		// all nodes take the other half
		for (; i<nArrangement && lists.size()>0; ) {
			int k = random.nextInt(lists.size());
			Node n = (Node)lists.get(k);
			if (n.number < MAX_DICE) {
				n.number++;
				i++;
				al.add(n);
			} else {
				lists.remove(k);
			}
		}		
		// return update nodes
		return extractUpdatedNodes(al);
	}
	// filter the duplicate nodes
	private Node[] extractUpdatedNodes(ArrayList al)
	{
		int i;
		boolean bIn[] = new boolean[nodes.length];
		for (i=0; i<al.size(); ) {
			Node n = (Node)al.get(i);
			if (bIn[n.index])
				al.remove(i);
			else {
				bIn[n.index] = true;
				i++;
			}
		}
		Node[] un = new Node[al.size()];
		for (i=0; i<un.length; i++) {
			un[i] = new Node((Node)al.get(i));
		}
		return un;
	}
	// return 0 if no one rules all nodes
	int isAllBelongsToOne()
	{
		for (int i=0; i<maxConnectedCount.length; i++)
			if (maxConnectedCount[i] == nNode)
				return i;
		return 0;
	}

	/**
	 * a attacks b
	 * @return return null if the attack is illegal, otherwise return the nodes modificated
	 */
	Node[] attack(int a, int b)
	{
		Node u = nodes[a];
		Node v = nodes[b];
		// check if argument is illegal
		if ((u.country!=v.country && u.number>=2 && edge[a][b] && u.country!=0 && v.country!=0) == false )
			return null;
		// battle!!
		int na = 0, nb = 0;
		for (int i=0; i<u.number; i++)
			na += random.nextInt(6) + 1;
		for (int i=0; i<v.number; i++)
			nb += random.nextInt(6) + 1;
		// return the nodes changed
		Node[] result;
		int un = u.number;
		u.number = 1;
		if (na < nb) {
			result = new Node[1];
			result[0] = new Node(u);
//			result[0] = u;
			return result;
		}
		v.number = un - 1;
		v.country = u.country;
		result = new Node[2];
		result[0] = new Node(u);
		result[1] = new Node(v);
//		result[0] = u;
//		result[1] = v;
		return result;
	}
	
	void setWidth(int n) { if (n >= 4 && n <= 12) width = n; }
	void setHeight(int n) { if (n >= 3 && n <= 8) height = n; }
	void setEmptyLevel(int n) { if (n >= 0 && n <= 20) emptyLevel = n; }
	void setNCountry(int n) { if (n >= 2 && n <= 8) nCountry = n; }
	void setDensityOfEdge(int n) { if (n >= 0 && n <= 100) densityOfEdge = n; }
}
